/* Copyright (C) 2012-2018 RealVNC Ltd. All Rights Reserved.
 *
 * This is sample code intended to demonstrate part of the
 * VNC Mobile Solution SDK. It is not intended as a production-ready
 * component.
 */

#include "UsbDeviceInfoProviderUnix.h"
#include "UsbDeviceInfoProvider.h"
#include <usbdeviceproviderdefs.h>
#include <assert.h>
#include <sys/wait.h>
#include <dirent.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <memory>
#include "UsbDevice.h"
#include <sys/time.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <list>
#include <string>
#include <sstream>
#include <net/if.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <errno.h>
#include <cstdlib>

extern "C"
{
#include <libudev.h>
}

/// The path to where devices are created by the kernel.
#define DEV_PATH "/dev/"
#define MAX_NET_DIR_SIZE 64

/// When receiving devices straight from the kernel, we need to allow the
/// system to configure the USB device. Therefore, it is good to have a small
/// delay (for example 10 ms) before doing anything with the device. If the
/// device is received from udev, instead of the kernel, there is no need to
/// have a delay (the device would be received after it is configured).
#define NEW_DEVICE_DELAY 10000
#ifdef USE_KERNEL_FOR_UDEV_MONITOR
#define NETLINK "kernel"
#else
#define NETLINK "udev"
#endif

#define NET_SUBSYSTEM "net"
#define USB_SUBSYSTEM "usb"

namespace
{
  /// Gets the name of the first subdir of a given dir within a given subpath.
  ///
  /// \param pPath The path in which to search for the parent dir.
  /// \param pParentDir The name of the dir that is the parent to the dir that
  /// is being searched for.
  /// \param pPrefix The prefix to be written together with the name. Must be
  /// non-NULL.
  /// \param depth The maximum depth to search. If negative, then there is no
  /// limit on the depth.
  /// \param pName The buffer where the name will be stored.
  /// \param nameSize The size of the buffer. Can store up to nameSize-1
  /// characters plus the terminating NUL character.
  ///
  /// \return 0 if not found, otherwise the number of characters + terminating
  /// NUL of he subdir name. If return value is greater than nameSize, then the
  /// buffer will be unchanged.
  size_t getSubdirName(const char* pPath, const char* pParentDir,
      const char* pPrefix,
      int depth, char* pName, size_t nameSize)
  {
    if(depth == 0)
      return 0;
    if(depth < -1)
      depth = -1; // if depth is negative, then just make it -1

    DIR *dir = opendir(pPath);
    if(!dir)
    {
      return 0;
    }

    // First check if the path contains the parent dir. If it does, then just
    // get the first subdir name.
    struct stat st;
    if(fstatat(dirfd(dir), pParentDir, &st, 0) >= 0 && (S_ISDIR(st.st_mode)))
    {
      try
      {
        std::string newPath = pPath;
        newPath += "/";
        newPath += pParentDir;

        DIR* targetDir = opendir(newPath.c_str());
        if(targetDir)
        {
          dirent* ent = readdir(targetDir);
          while(ent && ent->d_name[0] == '.')
          {
            ent = readdir(targetDir);
          }
          if(!ent)
          {
            closedir(targetDir);
          }
          else
          {
            // The name has been found. Write it and return.
            size_t size = strlen(pPrefix) + strlen(ent->d_name);
            if(size < nameSize)
            {
              strcpy(pName, pPrefix);
              strcat(pName, ent->d_name);
            }
            closedir(targetDir);
            closedir(dir);
            return size+1;
          }
        }
      }
      catch(std::bad_alloc&)
      { }
    }

    // List all the dirs in the current path and check each of them.  Decrease
    // the depth and iterate over a new path (old path + subdir name). Ignore
    // names starting with '.' throughout.
    try
    {
      dirent *dirEnt = NULL;
      while(NULL != (dirEnt = readdir(dir)))
      {
        if(dirEnt->d_name[0] == '.')
          continue;
        struct stat dirSt;
        if(fstatat(dirfd(dir), dirEnt->d_name, &dirSt, 0) < 0)
          continue;
        if(!S_ISDIR(dirSt.st_mode))
          continue;

        std::string newPath = pPath;
        newPath += "/";
        newPath += dirEnt->d_name;
        size_t size = getSubdirName(newPath.c_str(), pParentDir,
            pPrefix, depth-1, pName, nameSize);
        if(size > 0)
        {
          closedir(dir);
          return size;
        }
      }
    }
    catch(std::bad_alloc&)
    { }
    closedir(dir);
    return 0;
  }
}

using namespace vncdeviceprovider_usb;

UsbDeviceInfoObserver::~UsbDeviceInfoObserver()
{

}

UsbDeviceInfoProviderUnix::UsbDeviceInfoProviderUnix(
    VNCDeviceProviderBase& deviceProvider, UsbDeviceInfoObserver& observer)
  : mDeviceProvider(deviceProvider),
    mObserver(observer),
    mFd(VNCDiscovererNoEventHandle),
    mpUdev(NULL),
    mpMonitor(NULL),
    mpUdevEnum(NULL),
    mpEnumList(NULL)
{
}

UsbDeviceInfoProviderUnix::~UsbDeviceInfoProviderUnix()
{
#ifdef USE_KERNEL_FOR_UDEV_MONITOR
  mObserver.setTimerForUsbDeviceInfoProvider(VNCDiscoverySDKNoTimeout);
  for(UdevDevices::iterator it = mNewDevices.begin(); 
                      it != mNewDevices.end(); ++it)
  {
    delete (*it);
  }
  mNewDevices.clear();
#endif

  stopMonitoring();
  stopEnumeration();
  if(mpUdev)
  {
    udev_unref(mpUdev);
    mpUdev = NULL;
  }

  // check if any child processes have finished. Child processes are started by
  // runDhcpScript()
  waitpid(-1, NULL, WNOHANG);
}

bool UsbDeviceInfoProviderUnix::startMonitoring()
{
  // If already monitoring, then do nothing
  if(mpMonitor)
  {
    return true;
  }

  // Initialize udev if necessary
  if(!mpUdev)
  {
    mpUdev = udev_new();
    if(!mpUdev)
    {
      LOG_ERROR(&mDeviceProvider, "Unable to initialize Udev");
      return false;
    }
  }

  assert(!mpMonitor);

  // Initialize the udev monitor
  mpMonitor = udev_monitor_new_from_netlink(mpUdev, NETLINK);

  if(!mpMonitor)
  {
    LOG_ERROR(&mDeviceProvider, "Unable to initialize the Udev Monitor");
    return false;
  }

  // Add the filters
  vnc_int32_t err = udev_monitor_filter_add_match_subsystem_devtype(
                        mpMonitor, USB_SUBSYSTEM, NULL);
  if(err != 0)
  {
    LOG_ERROR_F(&mDeviceProvider, "Unable to set the 'usb' filter in the Udev"
          " Monitor, error %d", err);
    udev_monitor_unref(mpMonitor);
    mpMonitor = NULL;
    return false;
  }

  err = udev_monitor_filter_add_match_subsystem_devtype(
                        mpMonitor, NET_SUBSYSTEM, NULL);
  if(err != 0)
  {
    LOG_ERROR_F(&mDeviceProvider, "Unable to set the 'net' filter in the Udev"
          " Monitor, error %d", err);
    udev_monitor_unref(mpMonitor);
    mpMonitor = NULL;
    return false;
  }

  // Start the monitor
  if(0 != udev_monitor_enable_receiving(mpMonitor) ||
      VNCDiscovererNoEventHandle == (mFd = udev_monitor_get_fd(mpMonitor)))
  {
    LOG_ERROR(&mDeviceProvider, "Unable to connect Udev Monitor to the handle");
    udev_monitor_unref(mpMonitor);
    mpMonitor = NULL;
    return false;
  }

  // Reset any devices that might have been known from a previous run.
  resetExistingDevices();

  assert(mpMonitor);
  return true;
}

void UsbDeviceInfoProviderUnix::stopMonitoring()
{
  // check if any child processes have finished. Child processes are started by
  // runDhcpScript()
  waitpid(-1, NULL, WNOHANG);

  mFd = VNCDiscovererNoEventHandle;

  if(mpMonitor)
  {
    udev_monitor_unref(mpMonitor);
    mpMonitor = NULL;
  }
}

bool UsbDeviceInfoProviderUnix::startEnumeration()
{
  // Initialize udev if necessary
  if(!mpUdev)
  {
    mpUdev = udev_new();
    if(!mpUdev)
    {
      LOG_ERROR(&mDeviceProvider, "Unable to initialize Udev");
      return false;
    }
  }

  // Stop any existing enumeration
  stopEnumeration();

  assert(!mpEnumList);
  assert(!mpUdevEnum);

  checkNetInterfaces();

  // Instantiate the udev enumeration
  mpUdevEnum = udev_enumerate_new(mpUdev);

  if(!mpUdevEnum)
  {
    LOG_ERROR(&mDeviceProvider, "Enumeration was not initialized");
    return false;
  }

  // Add the filters (if any)
  vnc_int32_t err = udev_enumerate_add_match_subsystem(
      mpUdevEnum, USB_SUBSYSTEM);
  if(err != 0)
  {
    LOG_ERROR_F(&mDeviceProvider, "Unable to set the 'usb' filter in the Udev"
          " Enum, error %d", err);
    udev_enumerate_unref(mpUdevEnum);
    mpUdevEnum = NULL;
    return false;
  }

  err = udev_enumerate_scan_devices(mpUdevEnum);
  if(err != 0)
  {
    LOG_ERROR(&mDeviceProvider, "Unable to scan devices for enumeration");
    udev_enumerate_unref(mpUdevEnum);
    mpUdevEnum = NULL;
    return false;
  }

  mpEnumList = udev_enumerate_get_list_entry(mpUdevEnum);

  assert(mpUdevEnum);
  return true;
}

void UsbDeviceInfoProviderUnix::stopEnumeration()
{
  // check if any child processes have finished. Child processes are started by
  // runDhcpScript()
  waitpid(-1, NULL, WNOHANG);

  if(mpUdevEnum)
  {
    udev_enumerate_unref(mpUdevEnum);
    mpUdevEnum = NULL;
    mpEnumList = NULL;
  }
}

UsbDevice* UsbDeviceInfoProviderUnix::nextDevice()
{
  if(!mpEnumList)
    return NULL;

  assert(mpUdev);

  UsbDevice* pUsbDevice = NULL;
  while(mpEnumList)
  {
    const char* path = udev_list_entry_get_name(mpEnumList);
    mpEnumList = udev_list_entry_get_next(mpEnumList);
    if(!path)
    {
      LOG_ERROR(&mDeviceProvider, "NULL syspath in udev list");
      continue;
    }

    udev_device* pDev = udev_device_new_from_syspath(mpUdev, path);
    if(!pDev)
    {
      LOG_ERROR_F(&mDeviceProvider, "Unable to get device from syspath '%s'", path);
      continue;
    }

    vnc_uint8_t busNum, devNum;
    const char* pDevSysPath = NULL;
    const char* pSerial = NULL;
    if(!getDeviceDetails(pDev, busNum, devNum, pDevSysPath, pSerial))
    {
      udev_device_unref(pDev);
      continue;
    }

    pUsbDevice = new(std::nothrow) UsbDevice(busNum, devNum, path, pSerial);
    udev_device_unref(pDev);
    if(!pUsbDevice)
    {
      LOG_ERROR_F(&mDeviceProvider, "Out of memory allocating the USB device "
          "object for the device at syspath '%s'", path);
      continue;
    }

    // Add the device to the "known devices list" so it doesn't get
    // "re-discovered"
    checkDeviceActionIsNew(path, true);
    break;
  }

  return pUsbDevice;
}

void UsbDeviceInfoProviderUnix::handleEvent(VNCDiscovererEventHandle* pHandle)
{
  // check if any child processes have finished. Child processes are started by
  // runDhcpScript()
  waitpid(-1, NULL, WNOHANG);
  (void)pHandle;

  if(!mpMonitor)
    return;

  udev_device *pDev = udev_monitor_receive_device(mpMonitor);
  if(pDev)
  {
#ifdef LOG_DEVICE
    logDevice(pDev);
#endif

    const char* subsystem = udev_device_get_subsystem(pDev);
    // If this is a net subsytem update, then run the DHCP script, if one is
    // defined.
    if(subsystem && 0 == strcmp(subsystem, NET_SUBSYSTEM))
    {
      if(strcmp(udev_device_get_action(pDev), "add") == 0)
      {
        // Only run the dhcp script if it is a new interface.
        runDhcpScript(pDev);
      }
    }
    else
    {
      vnc_uint8_t busNum, devNum;
      const char* pSysPath = NULL;
      const char* pSerial = NULL;
      // Get the device details (like bus number, device number and syspath). If
      // the "device" received is of type "interface", then just ignore it.
      if(!getDeviceDetails(pDev, busNum, devNum, pSysPath, pSerial))
      {
        udev_device_unref(pDev);
        return;
      }

      if(strcmp(udev_device_get_action(pDev), "add") == 0)
      {
#ifdef USE_KERNEL_FOR_UDEV_MONITOR
        // When monitoring the kernel events directly, it's good to give a bit of
        // time to the OS to set up the device, before reporting.
        try
        {
          std::auto_ptr<UsbDevice> dev(new UsbDevice(busNum, devNum,
                pSysPath, pSerial));
          mNewDevices.push_back(dev.get());
          dev.release();
          mObserver.setTimerForUsbDeviceInfoProvider(NEW_DEVICE_DELAY);
        }
        catch(std::bad_alloc& e)
        {
          LOG_ERROR(&mDeviceProvider, "Out of memory when queuing new device for"
              " delay");
        }
#else
        UsbDevice usbDev(busNum, devNum, pSysPath);
        deviceAdded(usbDev);
#endif
      }
      else if(strcmp(udev_device_get_action(pDev), "remove") == 0)
      {
        deviceRemoved(busNum, devNum, pSysPath);
      }
    }

    if(pDev)
    {
      udev_device_unref(pDev);
    }
  }
  else 
  {
#ifdef USE_KERNEL_FOR_UDEV_MONITOR
    LOG_DEBUG(&mDeviceProvider, "No Device from receive_device().");
#else
    LOG_ERROR(&mDeviceProvider, "No Device from receive_device().");
#endif
  }   
}

void UsbDeviceInfoProviderUnix::handleTimeout()
{
  // check if any child processes have finished. Child processes are started by
  // runDhcpScript()
  waitpid(-1, NULL, WNOHANG);

#ifdef USE_KERNEL_FOR_UDEV_MONITOR
  // Handling all devices in one go. This shouldn't be a problem because of
  // a small timeout and the device connection rate is in general not very
  // high.
  for(UdevDevices::iterator it = mNewDevices.begin(); 
                      it != mNewDevices.end(); ++it)
  {
    deviceAdded(**it);
    delete (*it);
  }
  mNewDevices.clear();
#endif
}

VNCDiscoverySDKError UsbDeviceInfoProviderUnix::setProperty(const char *pProperty, const char *pValue)
{
  VNCDiscoverySDKError ret = VNCDiscoverySDKErrorNotSupported;
  if(strcmp(pProperty, VNCUsbDeviceProviderPropertyDhcpScript) == 0)
  {
    if(!pValue || pValue[0] == '\0')
    {
      mDhcpScriptName = "";
      ret = VNCDiscoverySDKErrorNone;
    }
    else
    {
      mDhcpScriptName = pValue;
      ret = VNCDiscoverySDKErrorNone;
    }
    checkNetInterfaces();
  }
  return ret;
}

const char* UsbDeviceInfoProviderUnix::getProperty(const char *pProperty)
{
  if(strcmp(pProperty, VNCUsbDeviceProviderPropertyDhcpScript) == 0)
  {
    return mDhcpScriptName.c_str();
  }
  return NULL;
}

size_t UsbDeviceInfoProviderUnix::getEventHandles(VNCDiscovererEventHandle *& pHandles)
{
  pHandles = &mFd;
  return 1;
}

size_t UsbDeviceInfoProviderUnix::getDevName(const char* pSysPath, vnc_uint8_t configurationValue,
    vnc_uint8_t interfaceNumber, vnc_uint8_t /* alternateSetting */,
    UsbDevice::InterfaceType interfaceType,
    char* pDevName, size_t devNameSize)
{
  try
  {
    // Get the interface path. First get the name of the device (from
    // '/sys//devices/pci0000:00/0000:00:1a.0/usb1/1-1/1-1.3/1-1.3.3' it's the
    // last bit: 1-1.3.3). Allowing for a terminating '/', go to the end of the
    // syspath and walk backwards until '/' is seen.
    const char* pName = strrchr(pSysPath, '\0');
    if(!pName)
      return 0;

    pName -= 2;
    while(pName > pSysPath && *pName != '/')
      --pName;

    if(pName == pSysPath)
      return 0;

    ++pName;
    std::ostringstream interfacePathStr;
    interfacePathStr.exceptions(std::ios_base::failbit | std::ios_base::badbit);

    // Compose the interface path like this:
    // 'sysPath/devName:confVal.ifNumber'.
    interfacePathStr << pSysPath << "/";
    interfacePathStr << pName << ":";
    interfacePathStr << (int)configurationValue;
    interfacePathStr << ".";
    interfacePathStr << (int)interfaceNumber;

    // Find in the interface path the subdir with the name given by the
    // interface type (either hidraw, or tty). Only go up to the maximum depth.
    // Expect to find something like
    // '/sys//devices/pci0000:00/0000:00:1a.0/usb1/1-1/1-1.3/1-1.3.3/1-1.3.3:1.1/../hidraw'
    return getSubdirName(interfacePathStr.str().c_str(),
        (interfaceType == UsbDevice::InterfaceTypeTty ? "tty" : "hidraw"),
        DEV_PATH, 5, pDevName, devNameSize);
  }
  catch(std::bad_alloc&)
  {
    return 0;
  }
  catch(std::ios_base::failure&)
  {
    return 0;
  }
  return 0;
}

bool UsbDeviceInfoProviderUnix::getDeviceDetails(udev_device* pDev, vnc_uint8_t& busNum,
    vnc_uint8_t& devNum, const char*& pSysPath, const char*& pSerial)
{
  const char* pDevType = udev_device_get_devtype(pDev);
  if(!pDevType || strcmp(pDevType, "usb_device") != 0)
  {
    // Only look at whole devices
    return false;
  }
  LOG_FLOW(&mDeviceProvider, "UsbDeviceInfoProviderUnix::getDeviceDetails");

  const char *const pPath = udev_device_get_syspath(pDev);

  // Get the details directly from udev. Checking sysfs would be a race
  // condition, as the device may already have disappeared (or been replaced).
  const char *const pBusNum = udev_device_get_property_value(pDev, "BUSNUM");
  const char *const pDevNum = udev_device_get_property_value(pDev, "DEVNUM");
  if(!pBusNum || !pPath || !pDevNum)
  {
    LOG_WARNING_F(&mDeviceProvider, "Unable to retrieve all details needed "
        "for device: path='%s', bus='%s', dev='%s'",
        (pPath ?  pPath : "(null)"),
        (pBusNum ?  pBusNum : "(null)"),
        (pDevNum ?  pDevNum : "(null)"));
    return false;
  }
  int bus = atoi(pBusNum);
  assert(bus <= 0xFF && bus >= 0);
  int dev = atoi(pDevNum);
  assert(dev <= 0xFF && dev >= 0);

  busNum = bus & 0xFF;
  devNum = dev & 0xFF;
  pSysPath = pPath;
  pSerial = udev_device_get_sysattr_value(pDev, "serial");
  if(!pSerial)
  {
    pSerial = udev_device_get_property_value(pDev, "ID_SERIAL_SHORT");
  }
  return true;
}

void UsbDeviceInfoProviderUnix::deviceAdded(const UsbDevice& dev)
{
  LOG_FLOW(&mDeviceProvider, "UsbDeviceInfoProviderUnix::deviceAdded");
  const char *sysPath = dev.sysPath();

  LOG_DEBUG_F(&mDeviceProvider, "Handling new device with sysPath '%s'", sysPath);

  if(!checkDeviceActionIsNew(sysPath, true))
  {
    LOG_DEBUG(&mDeviceProvider, "Device already known, ignoring");
    return;
  }

  mObserver.usbDeviceAdded(dev);
}

void UsbDeviceInfoProviderUnix::deviceRemoved(vnc_uint8_t& busNum,
    vnc_uint8_t devNum, const char*& pSysPath)
{
  if(!checkDeviceActionIsNew(pSysPath, false))
  {
    return;
  }

  mObserver.usbDeviceRemoved(UsbDevice::composeId(busNum, devNum));
}

bool UsbDeviceInfoProviderUnix::checkDeviceActionIsNew(const char* sysPath, bool isAdded)
{
#ifdef USE_KERNEL_FOR_UDEV_MONITOR
  try
  {
    if(isAdded)
    {
      if(mExistingDevices.find(sysPath) == mExistingDevices.end())
      {
        mExistingDevices.insert(sysPath);
        return true;
      }
    }
    else
    {
      // Need to remove from mNewDevices list if it's waiting to be
      // processed.
      for(UdevDevices::iterator it = mNewDevices.begin(); 
          it != mNewDevices.end(); ++it)
      {
        const char *sp = (*it)->sysPath();
        if(strcmp(sp, sysPath) == 0)
        {
          delete (*it);
          it = mNewDevices.erase(it);
          --it;
        }
      }

      DevicesPaths::iterator it = mExistingDevices.find(sysPath);
      if(it != mExistingDevices.end())
      {
        mExistingDevices.erase(it);
        return true;
      }
    }
  }
  catch(std::bad_alloc& e)
  {
      LOG_DEBUG_F(&mDeviceProvider, "Out of memory while checking syspath %s (added=%d)",
                    sysPath, (isAdded ? 1 : 0));
  }
  return false;
#else
  (void) sysPath;
  (void) isAdded;
  return true;
#endif
}

void UsbDeviceInfoProviderUnix::resetExistingDevices()
{
#ifdef USE_KERNEL_FOR_UDEV_MONITOR
  mExistingDevices.clear();
#endif
}

void UsbDeviceInfoProviderUnix::checkNetInterfaces()
{
  if(mDhcpScriptName.empty())
  {
    return;
  }

  // Instantiate the udev enumeration
  udev_enumerate* pEnum = udev_enumerate_new(mpUdev);

  if(!pEnum)
  {
    LOG_ERROR(&mDeviceProvider, "Net enumeration was not initialized");
    return;
  }

  // Add the net filter
  vnc_int32_t err = udev_enumerate_add_match_subsystem(
      pEnum, NET_SUBSYSTEM);
  if(err != 0)
  {
    LOG_ERROR_F(&mDeviceProvider, "Unable to set the 'net' filter in the Udev"
          " Enum, error %d", err);
    udev_enumerate_unref(pEnum);
    return;
  }

  // Scan for devices (network interfaces)
  err = udev_enumerate_scan_devices(pEnum);
  if(err != 0)
  {
    LOG_ERROR(&mDeviceProvider, "Unable to scan net devices for enumeration");
    udev_enumerate_unref(pEnum);
    return;
  }

  // Go through the list of interfaces and run the DHCP script if needed.
  udev_list_entry* pEnumList = udev_enumerate_get_list_entry(pEnum);
  while(pEnumList)
  {
    const char* path = udev_list_entry_get_name(pEnumList);
    pEnumList = udev_list_entry_get_next(pEnumList);
    if(!path)
    {
      LOG_ERROR(&mDeviceProvider, "NULL syspath in net udev list");
      continue;
    }

    udev_device* pDev = udev_device_new_from_syspath(mpUdev, path);
    if(!pDev)
    {
      LOG_ERROR_F(&mDeviceProvider, "Unable to get net device from syspath '%s'", path);
      continue;
    }
    runDhcpScript(pDev);
    udev_device_unref(pDev);
  }

  udev_enumerate_unref(pEnum);
}

void UsbDeviceInfoProviderUnix::runDhcpScript(udev_device *pDev)
{
  if(mDhcpScriptName.empty())
  {
    // Nothing to run
    return;
  }

  // The interface name is given by the system name. (Also, the interface name
  // can be found in the "INTERFACE" property).
  const char* pIfName = udev_device_get_sysname(pDev);
  if(!pIfName)
  {
    return;
  }

  // Check if the interface already has an IP address. If it does, then do
  // nothing.
  // Get a socket handle.
  int sck = socket(AF_INET, SOCK_DGRAM, 0);
  if(sck < 0)
  {
    LOG_ERROR_F(&mDeviceProvider, "socket returned error %d: %s", errno, strerror(errno));
    return;
  }

  // Get the available interfaces and iterate through them
  struct ifconf ifc;
  char buf[1024];
  ifc.ifc_len = sizeof(buf);
  ifc.ifc_buf = buf;
  if(ioctl(sck, SIOCGIFCONF, &ifc) < 0)
  {
    LOG_ERROR_F(&mDeviceProvider, "ioctl(SIOCGIFCONF) returned error %d: %s", errno, strerror(errno));
    close(sck);
    return;
  }

  struct ifreq *ifr = ifc.ifc_req;
  vnc_uint32_t nInterfaces = ifc.ifc_len / sizeof(struct ifreq);

  for(vnc_uint32_t i = 0; i < nInterfaces; i++)
  {
    struct ifreq *item = &ifr[i];

    if(0 == strcmp(pIfName, (const char *)item->ifr_name))
    {
      // Get the interface flags.
      if(ioctl(sck, SIOCGIFFLAGS, item) < 0)
      {
        LOG_DEBUG_F(&mDeviceProvider, "Unable to get flags for interface "
            "'%s', error %d: %s", pIfName, errno, strerror(errno));
      }
      short flags = item->ifr_flags;
      // Look only at interfaces that are not loopback, have a valid broadcast
      // address, support multicast, are down or have no IP address
      if((flags & IFF_LOOPBACK) != 0 || (flags & IFF_BROADCAST) == 0 ||
          (flags & IFF_MULTICAST) == 0 ||
          ((flags & IFF_UP) != 0 &&
           0 != (((struct sockaddr_in *)&item->ifr_addr)->sin_addr).s_addr))
      {
        close(sck);
        return;
      }
      // The interface needs to have DHCP run on it.
      break;
    }
  }

  close(sck);

  pid_t child_pid = fork();
  if(child_pid < 0)
  {
    LOG_ERROR_F(&mDeviceProvider, "Unable to create new process for running "
        "the DHCP script for interface '%s'. Error %d: %s",
        pIfName, errno, strerror(errno));
    return;
  }
  if(child_pid == 0)
  {
    const char * argv[3];

    argv[0] = mDhcpScriptName.c_str();
    argv[1] = pIfName;
    argv[2] = NULL;

    execvp(argv[0], (char * const *)argv);
    // If execvp returns then it failed to run the script
    std::exit(-errno);
  }
  else
  {
    LOG_DEBUG_F(&mDeviceProvider, "Running script: %s %s",
        mDhcpScriptName.c_str(), pIfName);
  }
}

#ifdef LOG_DEVICE
void UsbDeviceInfoProviderUnix::logDevice(udev_device *pDev)
{
  // Log what gets reported by the udev monitor
  // This is used to get device information to add it to the known
  // device types
  LOG_DEBUG(&mDeviceProvider, "Got Device");
  LOG_DEBUG_F(&mDeviceProvider, "   Node: %s", udev_device_get_devnode(pDev));
  LOG_DEBUG_F(&mDeviceProvider, "   DevPath: %s", udev_device_get_devpath(pDev));
  LOG_DEBUG_F(&mDeviceProvider, "   SysPath: %s", udev_device_get_syspath(pDev));
  LOG_DEBUG_F(&mDeviceProvider, "   SysName: %s", udev_device_get_sysname(pDev));
  LOG_DEBUG_F(&mDeviceProvider, "   SysNum: %s", udev_device_get_sysnum(pDev));
  LOG_DEBUG_F(&mDeviceProvider, "   Subsystem: %s", udev_device_get_subsystem(pDev));
  LOG_DEBUG_F(&mDeviceProvider, "   Devtype: %s", udev_device_get_devtype(pDev));
  LOG_DEBUG_F(&mDeviceProvider, "   Action: %s",udev_device_get_action(pDev));

  udev_list_entry *list = udev_device_get_properties_list_entry(pDev);
  LOG_DEBUG(&mDeviceProvider, "   Properties: ");
  while(list != NULL)
  {
    LOG_DEBUG_F(&mDeviceProvider, "      %s = %s", udev_list_entry_get_name(list),
            udev_list_entry_get_value(list));
    list = udev_list_entry_get_next(list);
  }
}
#endif

